#version 130
#extension GL_EXT_gpu_shader4 : enable
// the version and open GL extension
// should be the first line of the shader
/////////////////////////////////////////////////////////////////////////////////
// Computer Graphics - Task IIMod01.fsh  by   VNR
//https://www.shadertoy.com/view/7ltBzS
//Licence : Creative Commons Attribution-ShareAlike 4.0
//http://creativecommons.org/licences/by-sa/4.0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

#define iTime u_Elapsed*0.314159  //*0.1666
#define iResolution u_WindowSize

//#define mouse AUTO_MOUSE
//#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
//#define MOUSE_POS   vec2((1.0+cos(iTime*MOUSE_SPEED))*u_WindowSize/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
//#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )
//#define RIGID_SCROLL
// alternatively use static mouse definition
#define iMouse vec4(0.0,0.0, 0.0,0.0)
//#define iMouse vec4(512,256,180,120)
uniform sampler2D texture0;
uniform sampler2D texture1;
uniform sampler2D texture2;
uniform sampler2D texture3;
vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}
#define texture2D texture2D_Fract

#define MAX_STEPS      250
#define MAX_DIST       100.
#define EPSILON        0.01
#define HALF_CUBE_EDGE 1.
#define M_PI           3.1415926
#define F_DIEL         0.04
#define F_METAL        0.
#define GAMMA          2.2

const int MAT_CUBE  = 1;
const int MAT_PLANE = 2;

vec3 lightPos = vec3(0., 15., -10.);
vec3 rayOrigin = vec3(0., 1., -7.);
vec3 lookAtPoint = vec3(0.);
vec3 up = vec3(0., 1., 0.);

vec3 lightColor = vec3(1.);

const mat2 m2 = mat2(0.1, -1.2,
                     1.2,  0.1);

mat3 rotateX(float angle) {
    float c = cos(angle);
    float s = sin(angle);
    return mat3(
        vec3(1, 0,  0),
        vec3(0, c, -s),
        vec3(0, s,  c)
    );
}

mat3 rotateY(float angle) {
    float c = cos(angle);
    float s = sin(angle);
    return mat3(
        vec3( c, 0, s),
        vec3( 0, 1, 0),
        vec3(-s, 0, c)
    );
}

mat3 rotateZ(float angle) {
    float c = cos(angle);
    float s = sin(angle);
    return mat3(
        vec3(c, -s, 0),
        vec3(s,  c, 0),
        vec3(0,  0, 1)
    );
}

mat3 view(vec3 rayOrigin, vec3 lookAtPoint, vec3 up) {
    vec3 cameraZ = normalize(rayOrigin - lookAtPoint);
    vec3 cameraX = normalize(cross(up, cameraZ));
    vec3 cameraY = normalize(cross(cameraZ, cameraX));
    return mat3(cameraX, cameraY, cameraZ);
}

vec3 getTriplanarWeights(in vec3 n) {
    vec3 weights = pow(abs(n), vec3(2.));
    return weights / (weights.x + weights.y + weights.z);
}

float sdfRoundedBox(vec3 pos, vec3 size, float r) {
    vec3 q = abs (pos) - (size - r);
    return length(max(q, 0.0)) + min(max(q.x, max(q.y, q.z)), 0.0) - r;
}

float sdfTorus(vec3 pos, vec2 radii) {
    float x = length(pos.xz) - radii.x;
    return length(vec2(x, pos.y)) - radii.y;
}

float smoothUnion(float d1, float d2, float k) {
    float h = clamp( 0.5 + 0.5*(d2-d1)/k, 0.0, 1.0 );
    return mix( d2, d1, h ) - k*h*(1.0-h); 
}

float smoothSubtraction(float d1, float d2, float k) {
    float h = clamp( 0.5 - 0.5*(d2+d1)/k, 0.0, 1.0 );
    return mix( d2, -d1, h ) + k*h*(1.0-h); 
}

void rotateDice(inout vec3 pos) {
    pos *= rotateZ(iTime);
    pos *= rotateX(iTime);
}

float sdfDice(vec3 pos) {
    rotateDice(pos);    
    float boxDist = sdfRoundedBox(pos, vec3(HALF_CUBE_EDGE), 0.2);
    
    vec3 torusPosition = pos - vec3(0, HALF_CUBE_EDGE, 0.);
    float torusTop = sdfTorus(torusPosition, vec2(0.5, 0.125));
    
    torusPosition += vec3(0, 2.*HALF_CUBE_EDGE, 0.);
    float torusBottom = sdfTorus(torusPosition, vec2(0.5, 0.125));
    
    torusPosition -= vec3(0, 2.*HALF_CUBE_EDGE, 0.);
    torusPosition *= rotateZ(3.14/2.);
    torusPosition += vec3(-HALF_CUBE_EDGE, HALF_CUBE_EDGE, 0.);
    float torusLeft = sdfTorus(torusPosition, vec2(0.5, 0.125));
    
    torusPosition += vec3(0, -2.*HALF_CUBE_EDGE, 0.);
    float torusRight = sdfTorus(torusPosition, vec2(0.5, 0.125));
    
    torusPosition *= rotateX(3.14/2.);
    torusPosition += vec3(0, HALF_CUBE_EDGE, HALF_CUBE_EDGE);
    float torusFront = sdfTorus(torusPosition, vec2(0.5, 0.125));
    
    torusPosition += vec3(0, -2. * HALF_CUBE_EDGE, 0);
    float torusBack = sdfTorus(torusPosition, vec2(0.5, 0.125));
    
    float dist = smoothSubtraction(torusTop, boxDist, 0.1);
    dist = smoothSubtraction(torusBottom, dist, 0.1);
    dist = smoothSubtraction(torusLeft, dist, 0.1);
    dist = smoothSubtraction(torusRight, dist, 0.1);
    dist = smoothSubtraction(torusFront, dist, 0.1);
    dist = smoothSubtraction(torusBack, dist, 0.1);
    
    return dist;
}

float sdfHorizontalPlane(vec3 pos, float height) {
    return pos.y - height;
}

float sdfScene(vec3 pos) {
    float diceDist = sdfDice(pos);    
    float planeDist = sdfHorizontalPlane(pos, -2.*HALF_CUBE_EDGE);    
    float dist = min(diceDist, planeDist);
    
    return dist;
}

int getMaterial(vec3 pos) {
    float diceDist = sdfDice(pos);    
    float planeDist = sdfHorizontalPlane(pos, -2.*HALF_CUBE_EDGE); 
    float dist = min(diceDist, planeDist);
    
    int material = 0;    
    if (dist == diceDist)
        material = MAT_CUBE;
    else if (dist == planeDist)
        material = MAT_PLANE;    
    
    return material;
}

float rayMarching(in vec3 rayOrigin, in vec3 rayDirection, out bool hit) {
	vec3 point = rayOrigin;
	float totalDist = 0.;
	hit = false;

	for ( int steps = 0; steps < MAX_STEPS; steps++ ) {
		float dist = sdfScene(point);        
		if (dist < EPSILON) {
			hit = true;
			break;
		}		
		totalDist += 0.8 * dist;		
		if (totalDist > MAX_DIST)
			break;
		point += 0.8 * dist * rayDirection;
	}	
	return totalDist;
}

float ambientOcclusion(in vec3 pos, in vec3 normal) {
    float occ = 0.;
    float sca = 1.;
    for (int i = 0; i < 5; i++) {
        float h = 0.01 + 0.12 * float(i) / 4.;
        float d = sdfScene(pos + h * normal);
        occ += (h - d) * sca;
        sca *= 0.95;
        if (occ > 0.35) 
            break;
    }
    return clamp(1. - 3. * occ, 0., 1.)*(0.5 + 0.5 * normal.y);
}

vec3 fresnel(in vec3 f0, in float product) {
    product = clamp(product, 0., 1.);
    return mix(f0, vec3(1.), pow(1. - product, 5.));
}

float D_GGX(in vec3 n, in vec3 lightVec, in vec3 v, in float roughness) {
    float r2 = roughness * roughness;
    float r4 = r2 * r2;
    vec3 h = normalize(lightVec + v);
    float nh = dot(n, h);
    float nh2 = nh * nh;
    float d = (r4 - 1.) * nh2 + 1.;
    return r4 / (M_PI * d * d);
}

float G_neumann(in vec3 n, in vec3 lightVec, in vec3 v) {
    float nl = dot(n, lightVec);
    float nv = dot(n, v);
    return nl * nv/ max(nl, nv);
}

vec3 cookTorrance(in vec3 n, in vec3 lightVec, in vec3 v, in vec3 f0, in float roughness) {
    float D = D_GGX(n, lightVec, v, roughness);
    float G = G_neumann(n, lightVec, v);
    return f0 * D * G;
}

vec3 getNormal(vec3 pos, float accuracy) {
    float dist = sdfScene(pos);
    vec2 delta = vec2(accuracy / 2., 0); 
    vec3 n1 = vec3(
        sdfScene(pos + delta.xyy),
        sdfScene(pos + delta.yxy),
        sdfScene(pos + delta.yyx)
    );
    vec3 n2 = vec3(
        sdfScene(pos - delta.xyy),
        sdfScene(pos - delta.yxy),
        sdfScene(pos - delta.yyx)
    );
    return normalize(n1 - n2);
}

vec3 skyColor(in vec3 rayOrigin, in vec3 rayDirection) {
    vec3 color = vec3(0.5, 0.6, 1.) - 0.5 * rayDirection.y;    
    return color;
}

float fractionalBrownianMotion(vec2 point) {
    float f = 0.0;
    
    f += 0.5000 * texture2D(texture1, (point + 0.4 * iTime)/256.0).x;
    point = m2 * point * 2.02;
    
    f += 0.2500 * texture2D(texture1, (point + 0.1 * iTime)/256.0).x;
    point = m2 * point * 2.03;
    
    f += 0.1250 * texture2D(texture1, point / 256.0).x;
    point = m2 * point * 2.01;
    
    f += 0.0625 * texture2D(texture1, point/256.0).x;
    return f / 0.9375;
}

void ColorRoughnessMetalness(int material, in vec3 point,
                             inout vec3 baseColor, inout float roughness,
                             inout float metalness) {

    if (material == MAT_CUBE) {
        vec3 n1 = getNormal(point, 0.001);
        vec3 pos = point;
        rotateDice(n1); 
        rotateDice(pos); 
        vec3 tx = getTriplanarWeights (n1);
        vec2 textureScale = 6.0 * iResolution.xy/max(iResolution.x, iResolution.y);

        vec4 cx = texture2D(texture2, pos.yz / textureScale + 0.5);
        vec4 cy = texture2D(texture2, pos.zx / textureScale + 0.5);
        vec4 cz = texture2D(texture2, pos.xy / textureScale + 0.5);
        baseColor = vec4(cx * tx.x + cy * tx.y + cz * tx.z).xyz;

        float average = (baseColor.r + baseColor.g + baseColor.b) / 3.;
        roughness = baseColor.r - average;
        roughness = (roughness + 0.047058824) / 0.215686275;
        metalness = pow((1. - roughness), 2.);
        roughness *= 3.;
        baseColor = clamp(baseColor * 1.5, 0., 1.);
    }
    else if (material == MAT_PLANE) {
        baseColor = texture2D(texture0, point.xz * 0.25).rgb * vec3(1.5);
        metalness = 0.;
        roughness = 1.;
    }
}

bool isInShadow(in vec3 point, in vec3 lightVec, in vec3 n) {
    bool hit;
    float dist = rayMarching(point + n * EPSILON * 2., lightVec, hit);
    if (dist < length(lightPos - point))
        return true;
    return false;
}

vec3 calculateLighting(in vec3 baseColor, in float metalness, in float roughness,
                       in vec3 n, in vec3 v, in float nv, in vec3 lightVec, in float nl,
                       in vec3 point, in float dist, in int material) {
              
    baseColor = pow(baseColor, vec3(GAMMA));
    
    vec3 F0;
    if (material == MAT_CUBE)
        F0 = vec3(F_METAL);
    else if (material == MAT_PLANE)
        F0 = vec3(F_DIEL);
        
    F0 = mix(F0, baseColor, metalness);
    vec3 specFresnel = fresnel(F0, nv);
    vec3 spec = cookTorrance(n, lightVec, v, specFresnel, roughness)
        * nl / max(0.001, 4.* nl * nv);
    vec3 diff = (vec3(1.) - specFresnel) * nl / M_PI;
    vec3 color = (diff * mix(baseColor, vec3(0.), metalness) + spec) * lightColor;
    color += 0.25 * baseColor * ambientOcclusion(point, n);

    if (isInShadow(point, lightVec, n) && material == MAT_PLANE)
        color *= vec3(0.25, 0.3, 0.5);

    color = pow(color, vec3(1. / GAMMA));
    color = mix(color, vec3(0.5, 0.6, 1.0), pow(dist/MAX_DIST, 1.));
    
    return color;
}

vec3 getColor(in vec3 point, in float dist) {
    int material = getMaterial(point);
        
    vec3 lightVec = normalize(lightPos - point);
    vec3 n = getNormal(point, 0.001);
    vec3 v = normalize(rayOrigin - point);
    float nv = dot(n, v);
    float nl = max(0., dot(n, lightVec));

    vec3 baseColor;
    float metalness, roughness;

    ColorRoughnessMetalness(material, point, baseColor, roughness, metalness);  
    
    vec3 color = calculateLighting(baseColor, metalness, roughness, 
                                   n, v, nv, lightVec, nl,  point, dist, material);
    return color;
}

//void mainImage( out vec4 fragColor, in vec2 fragCoord )
///////////////////////////////////////////////////////////////////////////////// 
// need to convert this from a void to a function and call it by adding
// a void main(void) { to the end of the shader
// what type of variable will the function return?, it is a color and needs to be a vec4
// change void to vec4 
//void MainImage(out vec4 fragColor, in vec2 fragCoord) 
vec4 mainImage( out vec4 fragColor, in vec2 fragCoord )
{ 
    vec2 uv = (fragCoord - 0.5*iResolution.xy)/iResolution.y;
    vec2 mouse = iMouse.xy/iResolution.xy * 0.125 - 0.05;
    rayOrigin *= rotateX(-3.0 * mouse.y) * rotateY(-20.0 * mouse.x);
    rayOrigin *= rotateY(-iTime / (4. * M_PI));
    mat3 viewMatrix = view(rayOrigin, lookAtPoint, up);
    vec3 rayDirection = viewMatrix * normalize(vec3(uv, -1));
    bool hit;
    
    float dist = rayMarching(rayOrigin, rayDirection, hit);
    vec3 point = rayOrigin + rayDirection * dist;
    
    vec3 color = skyColor(rayOrigin, rayDirection);
    
    vec2 sc = rayOrigin.xz + rayDirection.xz * (250.*5000. - rayOrigin.y) / rayDirection.y;
    vec3 clouds = mix(color, vec3(1., 0.95, 1.),
        0.6 * smoothstep(0.4, 0.8, fractionalBrownianMotion(0.0001*sc/250.)));
    if (rayDirection.y > 0.)
        color = mix(color, clouds, clamp(rayDirection.y, 0., 0.01) * 100.);
      
    if (hit)
        color = getColor(point, dist);

    fragColor = vec4(color, 1.);
/////////////////////////////////////////////////////////////////////////////////
//the function needs to return a value. 
//it needs to be a vec4
//we will return the varable fragColor 
// usual place for fragColor = vec4( color, 1.0 ); bring the } down below
return fragColor; 
}

///////////////////////////////////////////////////////////////////////////////// 
void main(void) { // this will be run for every pixel of gl_FragCoord.xy
vec4 vTexCoord = gl_TexCoord[0];
vec4 fragColor = vec4(1.0); // initialize variable fragColor as a vec4 
vec4 cc = mainImage(fragColor, gl_FragCoord.xy); // call function mainImage and assign the return vec4 to cc
gl_FragColor = vec4(cc) * gl_Color; // set the pixel to the value of vec4 cc  and..
}

// ..uses the values of any Color: or Opacity:
// clauses (and any Animate clauses applied to these properties) 
// appearing in the Sprite, Quad or other node invoking the shader 
// in the .scn file.

